/*
 Copyright 2011 MYND-Ideal kft.
 Author: Barna Farago

 IdeState
 Used to parse a Paralel ATA / IDE interface states,
 and log to standard output as human readable list.

*/
#include <iostream>
#include <iomanip>
#include "idestate.h"

IdeState::IdeState():dataPtr(0), cylinder(0), busy(false), busyStart(0), busyMax(0), busyCount(0){

}
void IdeState::clear(){
    cs=3;
    addr=0; data=0;
    absTime=0;
}
void IdeState::process(){
    unsigned char code= cs<<4 |  addr;
    bool bNeedLog=true;
    bool bBuffLog=false;
     if (bRead){
         switch (code){
            case 0x27:
            case 0x16:
                errors.flags.ISE_wasErrorState|= (data &1) ; // "bit0:error ";
                if (!busy){
                    busy= (data &128)!=0;
                    if (busy) {
                        busyStart=absTime/1000; //in usec
                        busyCount++;
                    }
                }else{
                    busy= (data &128)!=0;
                    if (!busy){
                        signed long long dt= (absTime/1000) -(signed long long )busyStart;
                        if (busyMax<dt) busyMax=dt;
                        errors.flags.ISE_wasBusyState=true;
                    }
                }
                break;
            case 0x20:
                buff[dataPtr++]=data;
                bNeedLog=false;
                if (dataPtr>=MAXIDEBUFF){
                    dataPtr=0;  sectorCount++;
                    bNeedLog=true;
                    bBuffLog=true;
                }
                break;
        }
    }else{
        switch (code){
            case 0x16: //device control
                //if (data & 8)
                deviceControl=data;
                errors.flags.ISE_missingDevCtrl8|= !(data&8);
                errors.flags.ISE_missingDevCtrl1|= (data&1);
                break;
            case 0x26:
                drive= (data>>4)&1;
                head=data&0x0F;
                errors.flags.ISE_missingDriveA0= (data&0xA0)!=0xA0;
                break;
            case 0x25: cylinder=cylinder&0xFF | ((data&0xff)<<8); break;
            case 0x24: cylinder=cylinder&0xFF00 | (data&0xff); break;
            case 0x23: sector=data; break;
            case 0x22: sectorCount=data; break;
            case 0x20:
             buff[dataPtr++]=data;
             bNeedLog=false;
             if (dataPtr>=MAXIDEBUFF){
              dataPtr=0; sectorCount++;
              bNeedLog=true;
              bBuffLog=true;
             }
             break;
            case 0x27:
             lastCommand=data;
             switch (data){
                 case 0xc6: sectorMultiple=sectorCount; break; //setMultipleMode
             }
             break;
        }
    }
    if (bNeedLog) report();
    if (bBuffLog) {
        std::cout << "drive:" << drive;
        std::cout << " head:" << head;
        std::cout << " cylinder:" << cylinder;
        std::cout << " sector:" << sector;
        std::cout << " sectorCount:" << sectorCount-1;
        std::cout << " data:\n" ;
        for (int i=0; i<MAXIDEBUFF; i++){
            std::cout << "XX" << std::setw(2) << std::setfill('0')<< std::hex << buff[i] <<" ";
        }
        std::cout <<"\n";
    }
}
void IdeState::report(){
    std::cout << std::dec << absTime << ":"<< std::hex;
    if (bRead){
        std::cout << "read(";
    }else{
        std::cout << "write(";
    }
    unsigned char code= cs<<4 |  addr;

    std::cout<< cs
        <<","
        << addr
        <<","
        << data
        <<")";
    if (bRead){
         switch (code){
            case 0x27:
            case 0x16:
                std::cout<<"status(";
                if (data &1) std::cout<<"bit0:error ";
                if (data &2) std::cout<<"bit1:pulse ";
                if (data &4) std::cout<<"bit2:ecc ";
                if (data &8) std::cout<<"bit3:drq ";
                if (data &16) std::cout<<"bit4:skc ";
                if (data &32) std::cout<<"bit5:wtf ";
                if (data &64) std::cout<<"bit6:rdy ";
                if (data &128) std::cout<<"bit7:bsy ";
                std::cout<<")";
                break;
        }
    }else{
        switch (code){
            case 0x16:
                std::cout << "device_control(";
                if (deviceControl & 2) std::cout << "IRQdis "; else std::cout << "IRQen ";
                if (deviceControl & 4) std::cout << "swReset ";
                std::cout << ")";
             break;
            case 0x26: std::cout << "drive/head"; break;
            case 0x25: std::cout << "cylinder_high"; break;
            case 0x24: std::cout << "cylinder_low"; break;
            case 0x23: std::cout << "start_sector"; break;
            case 0x22: std::cout << "count_sector"; break;
            case 0x21: std::cout << "error_info"; break;
            case 0x20: std::cout << "data"; break;
            case 0x27:
                std::cout<<"command(";
                if ((data &0x0010)==0x0010) std::cout<<"recalibrate";
                if ((data &0x0070)==0x0070) std::cout<<"seek";
                switch(data){
                    case 0x03: std::cout<<"request_sense"; break;
                    case 0x20: case 0x21: std::cout<<"read_sector"; break;
                    case 0x22: case 0x23: std::cout<<"read_long_sector"; break;
                    case 0x30: case 0x31: std::cout<<"write_sector"; break;
                    case 0x32: case 0x33: std::cout<<"write_long_sector"; break;
                    case 0x38: std::cout<<"write_sector_noterase "; break;
                    case 0x3c: std::cout<<"write_verify "; break;
                    case 0x40: case 0x41: std::cout<<"verify_sector"; break;
                    case 0x50: std::cout<<"format_track "; break;
                    case 0x87: std::cout<<"translate_sector "; break;
                    case 0x90: std::cout<<"execute_drive_diagnostic "; break;
                    case 0x91: std::cout<<"init_drive_params "; break;
                    case 0x94: case 0xe0: std::cout<<"standby_immediate "; break;
                    case 0x95: case 0xe1: std::cout<<"idle_immediate "; break;
                    case 0x96: case 0xe2: std::cout<<"standby "; break;
                    case 0x97: case 0xe3: std::cout<<"idle "; break;
                    case 0x98: case 0xe5: std::cout<<"chk_power_mode "; break;
                    case 0x99: case 0xe6: std::cout<<"set_sleep_mode "; break;
                    case 0xc0: std::cout<<"erase_sector "; break;
                    case 0xc4: std::cout<<"multiple_read "; break;
                    case 0xc5: std::cout<<"multiple_write "; break;
                    case 0xc6: std::cout<<"set_multiple_mode"; break;
                    case 0xc8: std::cout<<"read_dma"; break;
                    case 0xca: std::cout<<"write_dma"; break;
                    case 0xcd: std::cout<<"multiple_write_noterase "; break;
                    case 0xe4: std::cout<<"read_buffer "; break;
                    case 0xe8: std::cout<<"write_buffer "; break;
                    case 0xec: std::cout<<"identify "; break;
                    case 0xef: std::cout<<"set_feature "; break;
                    case 0xf5: std::cout<<"wear_level "; break;
                    case 0xff: std::cout<<"nop "; break;
                }
                std::cout<<")";
                break;
        }
    }
    std::cout << "\n";
}
void IdeState::reportErrors(){
    if (!errors.dw) return;
    if (errors.flags.ISE_missingDriveA0){
        std::cout <<"drive/head parameter was sent but in wrong format\n";
    }
    if ((errors.flags.ISE_missingDevCtrl8)|| (errors.flags.ISE_missingDevCtrl1)){
        std::cout <<"device/control register was used but in wrong format\n";
    }
    if (errors.flags.ISE_wasBusyState){
        std::cout <<"divece was in busy "<< std::dec<<  busyCount <<"times, max aprox:" << busyMax << "usec\n";
    }
    if (errors.flags.ISE_wasErrorState){
        std::cout <<"divece was in error state\n";
    }
}
